/**
 * FAFO LABS Paper Trading - Background Service Worker
 * 
 * Handles:
 * - Auth state management (token storage)
 * - API request proxying (avoids CORS from content scripts)
 * - Badge updates (open position count)
 */

// Default backend URL (configurable via settings)
const DEFAULT_API_URL = 'https://fafolabs.xyz/api';

// ============ Storage Helpers ============

async function getApiUrl() {
    const { apiUrl } = await chrome.storage.local.get('apiUrl');
    return apiUrl || DEFAULT_API_URL;
}

async function getAuthToken() {
    const { authToken } = await chrome.storage.local.get('authToken');
    return authToken || null;
}

async function setAuthData(token, username) {
    if (token) {
        await chrome.storage.local.set({ authToken: token, username });
    } else {
        await chrome.storage.local.remove(['authToken', 'username']);
    }
    // Notify all tabs about auth state change
    broadcastToTabs({ type: 'AUTH_STATE_CHANGED', loggedIn: !!token, username });
}

async function getAuthState() {
    const data = await chrome.storage.local.get(['authToken', 'username']);
    return {
        loggedIn: !!data.authToken,
        username: data.username || null,
        token: data.authToken || null,
    };
}

// ============ API Request Handler ============

async function apiRequest(endpoint, options = {}) {
    const apiUrl = await getApiUrl();
    const token = await getAuthToken();

    const headers = {
        'Content-Type': 'application/json',
        ...options.headers,
    };

    if (token) {
        headers['Authorization'] = `Bearer ${token}`;
    }

    const url = `${apiUrl}${endpoint}`;

    try {
        const response = await fetch(url, {
            ...options,
            headers,
        });

        if (response.status === 401) {
            await setAuthData(null, null);
            return { error: 'Session expired. Please login again.', status: 401 };
        }

        const data = await response.json();

        if (!response.ok) {
            return { error: data.detail || 'API request failed', status: response.status };
        }

        return { data, status: response.status };
    } catch (err) {
        return { error: `Network error: ${err.message}`, status: 0 };
    }
}

// ============ Badge Updates ============

async function updateBadge() {
    const auth = await getAuthState();
    if (!auth.loggedIn) {
        chrome.action.setBadgeText({ text: '' });
        return;
    }

    const result = await apiRequest('/trades/open');
    if (result.data && Array.isArray(result.data)) {
        const count = result.data.length;
        chrome.action.setBadgeText({ text: count > 0 ? String(count) : '' });
        chrome.action.setBadgeBackgroundColor({ color: '#00e676' });
    }
}

// ============ Broadcast to Tabs & Side Panel ============

function broadcastToTabs(message) {
    chrome.tabs.query({}, (tabs) => {
        for (const tab of tabs) {
            chrome.tabs.sendMessage(tab.id, message).catch(() => { });
        }
    });
    // Also send to side panel (it listens via chrome.runtime.onMessage)
    // Side panel is same extension context, so it receives runtime messages directly
}

// ============ Message Handler ============

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    // All handlers are async, so we return true and use sendResponse
    handleMessage(message, sender).then(sendResponse);
    return true; // Keep the message channel open for async response
});

async function handleMessage(message, sender) {
    switch (message.type) {
        case 'LOGIN': {
            const result = await apiRequest('/auth/login', {
                method: 'POST',
                body: JSON.stringify({
                    username: message.username,
                    password: message.password,
                }),
            });

            if (result.data) {
                await setAuthData(result.data.access_token, result.data.username);
                updateBadge();
                return { success: true, data: result.data };
            }
            return { success: false, error: result.error };
        }

        case 'REGISTER': {
            const result = await apiRequest('/auth/register', {
                method: 'POST',
                body: JSON.stringify({
                    username: message.username,
                    password: message.password,
                    email: message.email || null,
                }),
            });

            if (result.data) {
                return { success: true, data: result.data };
            }
            return { success: false, error: result.error };
        }

        case 'LOGOUT': {
            await setAuthData(null, null);
            chrome.action.setBadgeText({ text: '' });
            return { success: true };
        }

        case 'GET_AUTH_STATE': {
            return await getAuthState();
        }

        case 'SET_API_URL': {
            await chrome.storage.local.set({ apiUrl: message.url });
            return { success: true };
        }

        case 'GET_API_URL': {
            return { url: await getApiUrl() };
        }

        case 'API_REQUEST': {
            const reqOptions = message.options ? {
                method: message.options.method || 'GET',
                body: message.options.body || undefined,
            } : {
                method: message.method || 'GET',
                body: message.body ? JSON.stringify(message.body) : undefined,
            };
            const result = await apiRequest(message.endpoint, reqOptions);

            if (result.error) {
                return { success: false, error: result.error };
            }
            return { success: true, data: result.data };
        }

        case 'SET_AUTH': {
            await setAuthData(message.token, message.username);
            broadcastToTabs({ type: 'AUTH_STATE_CHANGED', loggedIn: true, username: message.username });
            updateBadge();
            return { success: true };
        }



        case 'TOKEN_DETECTED': {
            // Relay token detection from content script to side panel
            // Side panel listens via chrome.runtime.onMessage
            // (message is already in runtime, side panel will receive it)
            return { success: true };
        }

        case 'UPDATE_BADGE': {
            await updateBadge();
            return { success: true };
        }

        case 'FETCH_PRICES': {
            // Fetch prices from Jupiter Price API v2 for given addresses
            const prices = {};
            const addresses = message.addresses || [];

            if (addresses.length > 0) {
                try {
                    // Jupiter supports batch: ?ids=mint1,mint2,mint3
                    const idsParam = addresses.join(',');
                    const response = await fetch(`https://api.jup.ag/price/v2?ids=${idsParam}`);
                    if (response.ok) {
                        const result = await response.json();
                        const data = result.data || {};
                        for (const addr of addresses) {
                            const tokenData = data[addr];
                            if (tokenData && tokenData.price) {
                                prices[addr] = parseFloat(tokenData.price);
                            }
                        }
                    }
                } catch (e) {
                    // Jupiter batch failed, try individually
                    for (const addr of addresses) {
                        try {
                            const response = await fetch(`https://api.jup.ag/price/v2?ids=${addr}`);
                            if (response.ok) {
                                const result = await response.json();
                                const tokenData = (result.data || {})[addr];
                                if (tokenData && tokenData.price) {
                                    prices[addr] = parseFloat(tokenData.price);
                                }
                            }
                        } catch (e2) {
                            // Ignore individual errors
                        }
                    }
                }
            }

            return { success: true, prices };
        }

        case 'SUBSCRIBE_PRICES': {
            const mints = message.mints || [];
            subscribeToPrices(mints);
            return { success: true, subscribed: mints };
        }

        default:
            return { error: 'Unknown message type' };
    }
}

// ============ Tab Navigation Listener ============

chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url) {
        // Send URL update to the content script
        chrome.tabs.sendMessage(tabId, {
            type: 'URL_CHANGED',
            url: tab.url,
        }).catch(() => { });
    }
});

// ============ Price WebSocket (Real-Time) ============

let priceWs = null;
let priceWsRetryDelay = 1000;
let subscribedMints = new Set();

function connectPriceWs() {
    if (priceWs) return;

    const wsUrl = 'wss://fafolabs.xyz/ws/prices';
    console.log('[FAFO BG] Connecting to price WS:', wsUrl);

    try {
        priceWs = new WebSocket(wsUrl);
    } catch (e) {
        console.error('[FAFO BG] WS creation failed:', e);
        scheduleReconnect();
        return;
    }

    priceWs.onopen = () => {
        console.log('[FAFO BG] Price WS connected');
        priceWsRetryDelay = 1000; // Reset backoff

        // Re-subscribe to all known mints
        if (subscribedMints.size > 0) {
            priceWs.send(JSON.stringify({
                type: 'subscribe',
                mints: Array.from(subscribedMints)
            }));
        }

        // Auto-subscribe to open position tokens
        autoSubscribeOpenPositions();
    };

    priceWs.onmessage = (event) => {
        try {
            const msg = JSON.parse(event.data);

            if (msg.type === 'price') {
                const priceMsg = {
                    type: 'PRICE_UPDATE',
                    mint: msg.mint,
                    price: msg.price,
                    source: msg.source,
                    ts: msg.ts
                };
                // Forward to all content scripts
                broadcastToTabs(priceMsg);
            }
        } catch (e) { }
    };

    priceWs.onclose = () => {
        console.log('[FAFO BG] Price WS disconnected');
        priceWs = null;
        scheduleReconnect();
    };

    priceWs.onerror = (err) => {
        console.error('[FAFO BG] Price WS error');
        priceWs = null;
    };
}

function scheduleReconnect() {
    setTimeout(() => {
        connectPriceWs();
    }, priceWsRetryDelay);
    // Exponential backoff, max 30 seconds
    priceWsRetryDelay = Math.min(priceWsRetryDelay * 1.5, 30000);
}

function subscribeToPrices(mints) {
    if (!Array.isArray(mints)) mints = [mints];
    for (const m of mints) {
        subscribedMints.add(m);
    }
    if (priceWs && priceWs.readyState === WebSocket.OPEN) {
        priceWs.send(JSON.stringify({
            type: 'subscribe',
            mints: mints
        }));
    }
}

async function autoSubscribeOpenPositions() {
    try {
        const result = await apiRequest('/trades/open');
        if (result.data && Array.isArray(result.data)) {
            const mints = result.data
                .map(t => t.token_address)
                .filter(Boolean);
            if (mints.length > 0) {
                subscribeToPrices(mints);
            }
        }
    } catch (e) {
        // Ignore
    }
}

// Connect price WS on startup
connectPriceWs();

// Update badge on startup
updateBadge();
