/**
 * FAFO LABS Paper Trading - Content Script
 * 
 * Injected into supported crypto trading platforms.
 * Detects current token, injects floating trading panel.
 */

(function () {
  'use strict';

  // Prevent double injection
  if (document.getElementById('fafo-paper-panel')) return;

  // ============ State ============
  let state = {
    loggedIn: false,
    username: null,
    balance: 0,
    detection: null,
    currentToken: null,
    openTrades: [],
    orders: [],
    history: [],
    activeTab: 'positions',
    currentPrice: null,
    quickAmounts: [25, 50, 100, 250, 500],
    editingQuickAmounts: false,
    activeTradeTab: 'MARKET', // MARKET, LIMIT, STOP
    tradeSide: 'BUY',         // BUY, SELL
    priceInput: '',           // For Limit/Stop prices
    lastSellAmount: '',       // For persisting sell input across re-renders
    watchlist: [], // Array of token objects
    collapsed: false,
    loading: true,
    statusMessage: null,
    panelMode: 'floating', // 'floating' or 'sidepanel'
    panelHidden: true, // Panel hidden by default, shown via extension icon
    // NEW: Real-time Data
    prices: {}, // Map<mint, price>
    ws: null,   // WebSocket instance
    token: null, // JWT Token for WS auth

    // UI State
    docked: false, // New: Dock to right side
  };

  // ============ Create Panel ============
  function createPanel() {
    const panel = document.createElement('div');
    panel.id = 'fafo-paper-panel';
    document.body.appendChild(panel);

    // Make draggable
    makeDraggable(panel);

    return panel;
  }

  function makeDraggable(panel) {
    let isDragging = false;
    let startX, startY, startLeft, startTop;

    panel.addEventListener('mousedown', (e) => {
      if (e.target.closest('.fafo-header') && !e.target.closest('button')) {
        isDragging = true;
        panel.classList.add('dragging');

        const rect = panel.getBoundingClientRect();
        startX = e.clientX;
        startY = e.clientY;
        startLeft = rect.left;
        startTop = rect.top;

        e.preventDefault();
      }
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;

      const dx = e.clientX - startX;
      const dy = e.clientY - startY;

      panel.style.left = (startLeft + dx) + 'px';
      panel.style.top = (startTop + dy) + 'px';
      panel.style.right = 'auto';
      panel.style.bottom = 'auto';
    });

    document.addEventListener('mouseup', () => {
      if (isDragging) {
        isDragging = false;
        panel.classList.remove('dragging');
      }
    });
  }

  // ============ Render ============
  function render() {
    const panel = document.getElementById('fafo-paper-panel');
    if (!panel) return;

    if (state.panelHidden) {
      panel.style.display = 'none';
      return;
    }
    panel.style.display = '';

    if (state.collapsed) {
      panel.classList.add('collapsed');
      panel.innerHTML = renderCollapsedHeader();
      attachEventListeners();
      return;
    }

    panel.classList.remove('collapsed');

    if (!state.loggedIn) {
      panel.innerHTML = renderHeader() + renderLoginPrompt();
      return;
    }

    panel.innerHTML =
      renderHeader() +
      '<div class="fafo-body">' +
      renderBalanceBar() +
      renderTokenBar() +
      renderQuickTrade() +
      renderTabs() +
      renderActiveTab() +
      renderStatus() +
      '</div>';

    attachEventListeners();
    updatePositionsPnl(); // Ensure PnL is calculated immediately after render
  }

  function renderCollapsedHeader() {
    return `
      <div class="fafo-header" data-action="drag">
        <div class="fafo-header-left">
          <div class="fafo-logo">F</div>
          <span class="fafo-brand">FAFO LABS</span>
        </div>
        <div class="fafo-header-right">
          <button class="fafo-btn-icon" data-action="expand-refresh" title="Expand & Refresh">⟳</button>
        </div>
      </div>
    `;
  }

  function renderHeader() {
    const platformBadge = state.detection
      ? `<span class="fafo-platform-badge">${state.detection.platformIcon} ${state.detection.platformName}</span>`
      : '';

    return `
      <div class="fafo-header" data-action="drag">
        <div class="fafo-header-left">
          <div class="fafo-logo">F</div>
          <span class="fafo-brand">FAFO LABS</span>
          ${platformBadge}
        </div>
        <div class="fafo-header-right">
          <button class="fafo-btn-icon" data-action="dock" title="${state.docked ? 'Undock' : 'Dock to Right'}">
            ${state.docked ? '⬕' : '⬔'}
          </button>
          <button class="fafo-btn-icon" data-action="refresh" title="Refresh">⟳</button>
          <button class="fafo-btn-icon" data-action="toggle" title="Collapse">▼</button>
          <button class="fafo-btn-icon" data-action="close-panel" title="Close" style="color:#ff5252;">✕</button>
        </div>
      </div>
    `;
  }

  function renderLoginPrompt() {
    return `
      <div class="fafo-login-prompt">
        <p>Login via the extension popup to start paper trading</p>
        <p style="font-size: 10px; color: #4a5568;">Click the FAFO icon in your browser toolbar</p>
      </div>
    `;
  }

  function calcTotalUnrealizedPnl() {
    let total = 0;
    for (const trade of state.openTrades) {
      const price = state.prices[trade.token_address];
      if (price && trade.entry_price && trade.quantity) {
        const currentValue = price * Number(trade.quantity);
        const costBasis = Number(trade.cost_usd || (Number(trade.entry_price) * Number(trade.quantity)));
        total += currentValue - costBasis;
      }
    }
    return total;
  }

  function renderBalanceBar() {
    const totalPnl = calcTotalUnrealizedPnl();
    const pnlClass = totalPnl >= 0 ? 'positive' : 'negative';

    return `
      <div class="fafo-balance-bar">
        <div>
          <div class="fafo-balance-label">Balance</div>
          <div class="fafo-balance-value">${Utils.formatUSD(state.balance)}</div>
        </div>
        <div style="text-align: right;">
          <div class="fafo-balance-label">Unrealized P&L</div>
          <div class="fafo-pnl ${pnlClass}" id="fafo-unrealized-pnl">${Utils.formatPnL(totalPnl)}</div>
        </div>
      </div>
    `;
  }

  function renderTokenBar() {
    if (!state.currentToken) {
      return `
        <div class="fafo-manual-token">
          <div class="fafo-section-title">Enter Token Address</div>
          <input type="text" class="fafo-input" id="fafo-manual-token-input" 
                 placeholder="Paste token address..." />
        </div>
      `;
    }

    const symbol = state.currentToken.symbol || '???';
    const address = state.currentToken.tokenAddress || '';
    const livePrice = state.currentPrice || state.prices[address] || null;

    return `
      <div class="fafo-token-bar">
        <div class="fafo-token-icon">🪙</div>
        <div class="fafo-token-info">
          <div class="fafo-token-symbol">${symbol}</div>
          <div class="fafo-token-address">${Utils.truncateAddress(address)}</div>
        </div>
        <div style="margin-left:auto; text-align:right;">
          ${livePrice ? `<div class="fafo-token-symbol" style="font-size:12px;">${Utils.formatPrice(livePrice)}</div>` : ''}
          <div class="fafo-token-status">● DETECTED</div>
        </div>
      </div>
    `;
  }

  function renderQuickTrade() {
    const tokenAddr = state.currentToken?.tokenAddress || '';
    const symbol = state.currentToken?.symbol || 'TOKEN';
    const disabled = !tokenAddr ? 'disabled' : '';
    const isBuy = state.tradeSide === 'BUY';
    const isLimit = state.activeTradeTab === 'LIMIT';
    const isStop = state.activeTradeTab === 'STOP';

    // Preview Setup
    let amountLabel = isBuy ? 'Amount ($)' : `Amount (${symbol})`;
    let previewHtml = '<div id="fafo-price-preview" class="fafo-preview-text">...</div>';

    // Quick Amounts
    let quickAmountsHtml = '';
    if (isBuy) {
      if (state.editingQuickAmounts) {
        quickAmountsHtml = `
                <div class="fafo-quick-amounts">
                    ${state.quickAmounts.map((amt, i) => `
                        <input type="number" class="fafo-quick-amount-input" data-index="${i}" value="${amt}">
                    `).join('')}
                </div>
                <button class="fafo-btn-secondary" data-action="save-quick-amounts" style="width:100%; margin-top:8px; padding:4px;">Save Presets</button>
            `;
      } else {
        quickAmountsHtml = `
                <div class="fafo-quick-amounts">
                    <div class="fafo-edit-link" data-action="toggle-edit-amounts">Edit</div>
                    ${state.quickAmounts.map(amt => `
                        <button class="fafo-quick-amount" data-amount="${amt}">$${amt}</button>
                    `).join('')}
                </div>
            `;
      }
    } else {
      // Sell Percentages
      quickAmountsHtml = `
            <div class="fafo-quick-amounts">
                <button class="fafo-quick-amount" data-pct="0.25">25%</button>
                <button class="fafo-quick-amount" data-pct="0.50">50%</button>
                <button class="fafo-quick-amount" data-pct="0.75">75%</button>
                <button class="fafo-quick-amount" data-pct="1.00">100%</button>
            </div>
        `;
    }

    return `
      <div class="fafo-trade-section">
        <!-- Side Toggle -->
        <div style="display: flex; gap: 8px; margin-bottom: 8px;">
            <button class="fafo-btn-toggle ${isBuy ? 'active-buy' : ''}" data-side="BUY" style="flex:1; border-radius:4px; padding:6px; border:none; ${isBuy ? 'background:#00e676; color:#000; font-weight:700;' : 'background:rgba(255,255,255,0.1); color:#8a9bb0;'}">BUY</button>
            <button class="fafo-btn-toggle ${!isBuy ? 'active-sell' : ''}" data-side="SELL" style="flex:1; border-radius:4px; padding:6px; border:none; ${!isBuy ? 'background:#ff5252; color:#fff; font-weight:700;' : 'background:rgba(255,255,255,0.1); color:#8a9bb0;'}">SELL</button>
        </div>

        <!-- Type Tabs -->
        <div class="fafo-trade-tabs">
            <button class="fafo-trade-tab ${state.activeTradeTab === 'MARKET' ? 'active' : ''}" data-trade-tab="MARKET">Market</button>
            <button class="fafo-trade-tab ${isLimit ? 'active' : ''}" data-trade-tab="LIMIT">Limit</button>
            ${!isBuy ? `<button class="fafo-trade-tab ${isStop ? 'active' : ''}" data-trade-tab="STOP">Stop</button>` : ''}
        </div>

        <!-- Inputs -->
        <div class="fafo-input-group">
            ${(isLimit || isStop) ? `
            <div class="fafo-input-row">
                <div style="flex:1">
                    <label class="fafo-input-label">${isStop ? 'Trigger Price' : 'Price'} ($)</label>
                    <input type="number" class="fafo-input" id="fafo-trade-price" 
                           placeholder="Price" step="0.00000001" value="${state.priceInput || state.currentPrice || state.prices[tokenAddr] || ''}" />
                </div>
            </div>` : ''}
            
            <div class="fafo-input-row">
                <div style="flex:1">
                    <label class="fafo-input-label">${amountLabel}</label>
                    <input type="number" class="fafo-input" id="fafo-trade-amount" 
                           placeholder="${isBuy ? 'USD Amount' : 'Token Amount'}" min="0" step="any" value="${isBuy ? (state.lastBuyAmount || '100') : (state.lastSellAmount || '')}" />
                    ${previewHtml}
                </div>
            </div>
        </div>

        <!-- Action Button -->
        <button class="fafo-btn-primary ${!isBuy ? 'fafo-btn-sell' : ''}" id="fafo-action-btn" ${disabled} 
                style="width: 100%; margin-top: 4px; font-weight: 800; padding: 10px;">
            ${isBuy ? 'BUY' : 'SELL'} ${symbol} ${isLimit ? 'LIMIT' : (isStop ? 'STOP' : '')}
        </button>

        <!-- Quick Amounts -->
        ${quickAmountsHtml}
      </div>
    `;
  }

  function renderTabs() {
    const tabs = [
      { id: 'positions', label: `Positions (${state.openTrades.length})` },
      { id: 'orders', label: `Orders (${state.orders.length})` },
      { id: 'history', label: 'History' },
      { id: 'watchlist', label: `Watch (${state.watchlist.length})` },
    ];

    return `
      <div class="fafo-tabs">
        ${tabs.map(t => `
          <button class="fafo-tab ${state.activeTab === t.id ? 'active' : ''}" 
                  data-tab="${t.id}">${t.label}</button>
        `).join('')}
      </div>
    `;
  }

  function renderActiveTab() {
    switch (state.activeTab) {
      case 'positions': return renderPositions();
      case 'orders': return renderOrders();
      case 'history': return renderHistory();
      case 'watchlist': return renderWatchlist();
      default: return '';
    }
  }

  function renderPositions() {
    if (state.openTrades.length === 0) {
      return `
        <div class="fafo-positions-section">
          <div class="fafo-empty">
            <div class="fafo-empty-icon">📭</div>
            <div>No open positions</div>
          </div>
        </div>
      `;
    }

    const getTokenLink = (addr, sym) => {
      return `<a href="#" class="fafo-position-symbol-link" data-action="navigate" data-token-address="${addr}" data-symbol="${sym}">${sym}</a>`;
    }

    const items = state.openTrades.map(trade => {
      let pnl = Number(trade.pnl || 0);

      // Calculate live PnL if we have a price
      let price = state.prices[trade.token_address];

      // Fallback to current page price if matches
      if (!price && state.currentToken && state.currentPrice) {
        if (state.currentToken.tokenAddress === trade.token_address) {
          price = state.currentPrice;
        }
      }

      if (price) {
        const entry = Number(trade.entry_price);
        const size = Number(trade.position_size);
        pnl = (price - entry) * size;
      }

      const pnlClass = pnl >= 0 ? 'positive' : 'negative';
      const pnlPct = Utils.pnlPercent(trade.entry_price, Number(trade.entry_price) + pnl / Number(trade.position_size));

      return `
        <div class="fafo-position-item">
          <div class="fafo-position-left">
            <span class="fafo-position-symbol">${getTokenLink(trade.token_address, trade.symbol)}</span>
            <span class="fafo-position-entry">
              Entry: ${Utils.formatPrice(trade.entry_price)} · ${Utils.formatUSD(trade.cost_usd)}
            </span>
          </div>
          <div class="fafo-position-right">
            <div class="fafo-position-pnl">
              <div class="fafo-position-pnl-value fafo-pnl ${pnlClass}" 
                   data-pnl-trade-id="${trade.trade_id}" 
                   data-entry="${trade.entry_price}" 
                   data-size="${trade.position_size}">
                ${Utils.formatPnL(pnl)}
              </div>
              <div class="fafo-position-pnl-amount" data-pct-trade-id="${trade.trade_id}">
                ${Utils.formatPercent(pnlPct)}
              </div>
            </div>
            <button class="fafo-btn-close" data-close-trade="${trade.trade_id}">CLOSE</button>
          </div>
        </div>
      `;
    }).join('');

    return `<div class="fafo-positions-section">${items}</div>`;
  }

  function renderOrders() {
    if (state.orders.length === 0) {
      return `
        <div class="fafo-orders-section">
          <div class="fafo-empty">
            <div class="fafo-empty-icon">📋</div>
            <div>No pending orders</div>
          </div>
        </div>
      `;
    }

    const getTokenLink = (addr, sym) => {
      let url = `https://dexscreener.com/solana/${addr}`;
      if (state.detection && state.detection.getTokenUrl) {
        url = state.detection.getTokenUrl(addr);
      }
      return `<a href="${url}" class="fafo-position-symbol-link" target="_blank">${sym}</a>`;
    }

    const items = state.orders.map(order => {
      const typeClass = order.order_type.toLowerCase().replace('_', '-');
      return `
        <div class="fafo-order-item">
          <div>
            <span class="fafo-order-type ${typeClass}">${order.order_type}</span>
            <span style="margin-left: 6px; font-weight: 600;">${getTokenLink(order.token_address, order.symbol)}</span>
          </div>
          <div style="display: flex; align-items: center; gap: 8px;">
            <span style="font-size: 11px; color: #8a9bb0;">
              ${order.side} @ ${Utils.formatPrice(order.target_price)}
            </span>
            <button class="fafo-btn-cancel" data-cancel-order="${order.order_id}">✕</button>
          </div>
        </div>
      `;
    }).join('');

    return `<div class="fafo-orders-section">${items}</div>`;
  }

  function renderHistory() {
    if (state.history.length === 0) {
      return `
        <div class="fafo-history-section">
          <div class="fafo-empty">
            <div class="fafo-empty-icon">📜</div>
            <div>No trade history</div>
          </div>
        </div>
      `;
    }

    const getTokenLink = (addr, sym) => {
      let url = `https://dexscreener.com/solana/${addr}`;
      if (state.detection && state.detection.getTokenUrl) {
        url = state.detection.getTokenUrl(addr);
      }
      return `<a href="${url}" class="fafo-position-symbol-link" target="_blank">${sym}</a>`;
    }

    const items = state.history.slice(0, 10).map(trade => {
      const pnl = Number(trade.pnl || 0);
      const pnlClass = pnl >= 0 ? 'positive' : 'negative';
      const entryMcap = Utils.formatMCap(trade.entry_mcap);
      const exitMcap = Utils.formatMCap(trade.exit_mcap);

      return `
        <div class="fafo-history-item">
          <div>
            <span class="fafo-history-symbol">${getTokenLink(trade.token_address, trade.symbol)}</span>
            <span class="fafo-history-time">${Utils.timeAgo(trade.closed_at)}</span>
            ${entryMcap || exitMcap ? `
            <div style="font-size:9px; color:#6b7a8d; margin-top:2px;">
              ${entryMcap ? `<span style="color:#8a9bb0;">Buy MC: <span style="color:#00e676;">${entryMcap}</span></span>` : ''}
              ${entryMcap && exitMcap ? ' → ' : ''}
              ${exitMcap ? `<span style="color:#8a9bb0;">Sell MC: <span style="color:#ff5252;">${exitMcap}</span></span>` : ''}
            </div>` : ''}
          </div>
          <span class="fafo-pnl ${pnlClass}">${Utils.formatPnL(pnl)}</span>
        </div>
      `;
    }).join('');

    return `<div class="fafo-history-section">${items}</div>`;
  }

  function renderStatus() {
    if (!state.statusMessage) return '';
    const cls = state.statusMessage.type || 'success';
    return `<div class="fafo-status ${cls}">${state.statusMessage.text}</div>`;
  }

  function renderWatchlist() {
    const currentAddr = state.currentToken?.tokenAddress;

    let html = '<div class="fafo-watchlist-section">';

    if (state.currentToken?.tokenAddress) {
      const isWatched = state.watchlist.some(t => t.address.toLowerCase() === currentAddr.toLowerCase());
      if (!isWatched) {
        html += `
                <div style="padding: 8px;">
                    <button class="fafo-btn-secondary" id="fafo-add-watch" style="width:100%; margin-bottom:8px;">+ Watch ${state.currentToken.symbol}</button>
                </div>`;
      }
    }

    if (state.watchlist.length === 0) {
      html += `<div class="fafo-empty"><div class="fafo-empty-icon">⭐</div><div>No watched tokens</div></div>`;
    } else {
      const items = state.watchlist.map(t => {
        let url = '#';
        if (state.detection && state.detection.getTokenUrl) {
          url = state.detection.getTokenUrl(t.address);
        } else {
          url = `https://dexscreener.com/solana/${t.address}`;
        }

        return `
                <div class="fafo-list-item" style="padding: 8px; border-bottom: 1px solid rgba(255,255,255,0.1); display: flex; align-items: center; justify-content: space-between;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <button class="fafo-btn-icon" data-remove-watch="${t.address}" title="Remove">★</button>
                        <div>
                            <a href="${url}" class="fafo-position-symbol-link" target="_blank">${t.symbol}</a>
                            <div style="font-size:10px; color:#6b7a8d;">${Utils.truncateAddress(t.address)}</div>
                        </div>
                    </div>
                </div>`;
      }).join('');
      html += items;
    }

    html += '</div>';
    return html;
  }

  // ============ Event Listeners ============
  function attachEventListeners() {
    const panel = document.getElementById('fafo-paper-panel');
    if (!panel) return;

    // Toggle collapse
    panel.querySelectorAll('[data-action="toggle"]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        state.collapsed = !state.collapsed;
        render();
      });
    });

    // Close panel (hide)
    panel.querySelectorAll('[data-action="close-panel"]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        state.panelHidden = true;
        render();
      });
    });

    // Refresh
    panel.querySelectorAll('[data-action="refresh"]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        refreshData();
      });
    });

    // Expand + Refresh (collapsed state)
    panel.querySelectorAll('[data-action="expand-refresh"]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        state.collapsed = false;
        render();
        refreshData();
      });
    });




    // Dock / Undock
    panel.querySelectorAll('[data-action="dock"]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        state.docked = !state.docked;
        const p = document.getElementById('fafo-paper-panel');
        if (p) {
          if (state.docked) p.classList.add('docked');
          else p.classList.remove('docked');
        }
        render(); // Update icon
      });
    });

    // Tab switching
    panel.querySelectorAll('[data-tab]').forEach(tab => {
      tab.addEventListener('click', () => {
        state.activeTab = tab.dataset.tab;
        render();
      });
    });

    // Side switching (Buy/Sell)
    panel.querySelectorAll('[data-side]').forEach(btn => {
      btn.addEventListener('click', () => {
        state.tradeSide = btn.dataset.side;
        // Reset to MARKET if switching to BUY while STOP is active (Stop is sell-only)
        if (state.tradeSide === 'BUY' && state.activeTradeTab === 'STOP') {
          state.activeTradeTab = 'MARKET';
        }
        render();
      });
    });

    // Trade Tab switching (Market/Limit/Stop)
    panel.querySelectorAll('[data-trade-tab]').forEach(btn => {
      btn.addEventListener('click', () => {
        state.activeTradeTab = btn.dataset.tradeTab;
        state.priceInput = ''; // Reset so current price auto-fills
        render();
      });
    });

    // Price Input
    const priceInput = document.getElementById('fafo-trade-price');
    if (priceInput) {
      priceInput.addEventListener('input', (e) => {
        state.priceInput = e.target.value;
      });
    }

    // Amount Input
    const amountInput = document.getElementById('fafo-trade-amount');
    if (amountInput) {
      amountInput.addEventListener('input', (e) => {
        if (state.tradeSide === 'BUY') {
          state.lastBuyAmount = e.target.value;
        } else {
          state.lastSellAmount = e.target.value;
        }
        updatePricePreview();
      });
      updatePricePreview();
    }

    // Quick amounts (Buy: Value, Sell: Pct)
    panel.querySelectorAll('.fafo-quick-amount').forEach(btn => {
      btn.addEventListener('click', () => {
        if (state.tradeSide === 'BUY') {
          const val = btn.dataset.amount;
          state.lastBuyAmount = val;
          const input = document.getElementById('fafo-trade-amount');
          if (input) {
            input.value = val;
            updatePricePreview();
          }
        } else {
          // Sell Percentage
          const pct = parseFloat(btn.dataset.pct);
          const trade = findTradeForCurrentToken();
          if (trade) {
            const input = document.getElementById('fafo-trade-amount');
            if (input) {
              const amt = (trade.position_size * pct).toFixed(6);
              input.value = amt;
              state.lastSellAmount = amt;
              updatePricePreview();
            }
          }
        }
      });
    });

    // Edit Quick Amounts Toggles
    const editLink = panel.querySelector('[data-action="toggle-edit-amounts"]');
    if (editLink) {
      editLink.addEventListener('click', () => {
        state.editingQuickAmounts = !state.editingQuickAmounts;
        render();
      });
    }

    // Position Navigation
    panel.querySelectorAll('[data-action="navigate"]').forEach(link => {
      link.addEventListener('click', (e) => {
        e.preventDefault();
        const addr = link.dataset.tokenAddress;
        const sym = link.dataset.symbol;
        console.log('[FAFO] Navigate:', addr, sym, state.detection);
        if (typeof PlatformDetector !== 'undefined' && state.detection) {
          PlatformDetector.navigate(state.detection, addr, sym);
        } else {
          // Fallback
          window.location.href = `https://dexscreener.com/solana/${addr}`;
        }
      });
    });

    const savePresetsBtn = panel.querySelector('[data-action="save-quick-amounts"]');
    if (savePresetsBtn) {
      savePresetsBtn.addEventListener('click', () => {
        const inputs = panel.querySelectorAll('.fafo-quick-amount-input');
        const newAmounts = [];
        inputs.forEach(inp => {
          const val = parseInt(inp.value);
          if (!isNaN(val) && val > 0) newAmounts.push(val);
        });
        if (newAmounts.length === 5) {
          state.quickAmounts = newAmounts;
          state.editingQuickAmounts = false;
          if (typeof chrome !== 'undefined' && chrome.storage) {
            chrome.storage.local.set({ quickAmounts: newAmounts });
          }
          render();
        } else {
          showStatus('Must have 5 positive values', 'error');
        }
      });
    }

    // Main Action Button (Buy/Sell/Order)
    const actionBtn = document.getElementById('fafo-action-btn');
    if (actionBtn) {
      actionBtn.addEventListener('click', handleTradeAction);
    }

    // Close trade buttons
    panel.querySelectorAll('[data-close-trade]').forEach(btn => {
      btn.addEventListener('click', () => handleCloseTrade(btn.dataset.closeTrade));
    });

    // Cancel order buttons
    panel.querySelectorAll('[data-cancel-order]').forEach(btn => {
      btn.addEventListener('click', () => handleCancelOrder(btn.dataset.cancelOrder));
    });

    // Watchlist Add
    const addWatchBtn = document.getElementById('fafo-add-watch');
    if (addWatchBtn) {
      addWatchBtn.addEventListener('click', () => {
        if (state.currentToken && state.currentToken.tokenAddress) {
          // Check if already exists
          if (!state.watchlist.some(t => t.address === state.currentToken.tokenAddress)) {
            state.watchlist.push({
              address: state.currentToken.tokenAddress,
              symbol: state.currentToken.symbol,
              chain: state.currentToken.chain || 'solana'
            });
            saveWatchlist();
            render();
          }
        }
      });
    }

    // Watchlist Remove
    panel.querySelectorAll('[data-remove-watch]').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.stopPropagation();
        const addr = btn.dataset.removeWatch;
        state.watchlist = state.watchlist.filter(t => t.address !== addr);
        saveWatchlist();
        render();
      });
    });

    // Manual token input
    const manualInput = document.getElementById('fafo-manual-token-input');
    if (manualInput) {
      manualInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter') {
          const address = manualInput.value.trim();
          if (address) {
            state.currentToken = {
              tokenAddress: address,
              symbol: address.slice(0, 6).toUpperCase(),
              chain: 'solana',
            };
            render();
          }
        }
      });
    }
  }

  // ============ Pre-trade Token Address Resolution ============
  async function resolveTokenAddressIfNeeded() {
    if (!state.detection || !state.currentToken?.tokenAddress) return;

    const addr = state.currentToken.tokenAddress;

    // If detection has a resolve function (DexScreener), try to resolve pair → mint
    // This ensures we have the actual token mint, not a pair/pool address
    if (state.detection.resolve) {
      // First try DOM-based extraction (fastest, no network)
      if (state.detection.getTokenMintFromPage) {
        const domMint = state.detection.getTokenMintFromPage();
        if (domMint && domMint !== addr && domMint.length >= 32) {
          console.log('[FAFO] Pre-trade: resolved via DOM to', domMint);
          state.currentToken.tokenAddress = domMint;
          return;
        }
      }

      // Then try API-based resolution
      try {
        const resolved = await state.detection.resolve(addr);
        if (resolved && resolved !== addr) {
          console.log('[FAFO] Pre-trade: resolved via API to', resolved);
          state.currentToken.tokenAddress = resolved;
        }
      } catch (e) {
        console.warn('[FAFO] Pre-trade resolution failed:', e);
      }
    }
  }

  // ============ Actions ============
  async function handleTradeAction() {
    if (!state.currentToken?.tokenAddress) {
      showStatus('No token detected.', 'error');
      return;
    }

    const amountInput = document.getElementById('fafo-trade-amount');
    // For Sell, amount is tokens. For Buy, amount is USD.
    const amountVal = parseFloat(amountInput?.value);

    // Basic validation
    if (!amountVal || amountVal <= 0) {
      showStatus('Enter a valid amount', 'error');
      return;
    }

    const btn = document.getElementById('fafo-action-btn');
    if (btn) {
      btn.disabled = true;
      btn.textContent = 'PROCESSING...';
    }

    try {
      // Pre-trade: ensure token address is resolved (DexScreener pair → mint)
      await resolveTokenAddressIfNeeded();

      // MARKET BUY
      if (state.activeTradeTab === 'MARKET' && state.tradeSide === 'BUY') {
        const result = await API.openTrade(
          state.currentToken.symbol || 'UNKNOWN',
          state.currentToken.tokenAddress,
          amountVal // USD
        );
        // Use execution price if available (for aggregated trades), otherwise entry price
        const price = result.execution_price || result.entry_price;
        showStatus(`Bought ${result.symbol} @ ${Utils.formatPrice(price)}`, 'success');
      }
      // MARKET SELL (Close Position / Partial)
      else if (state.activeTradeTab === 'MARKET' && state.tradeSide === 'SELL') {
        await closePositionForToken(amountVal); // Tokens
      }
      // LIMIT / STOP ORDERS
      else {
        // Read price from DOM input first (handles auto-fill), then state fallbacks
        const priceEl = document.getElementById('fafo-trade-price');
        const price = parseFloat(priceEl?.value) || parseFloat(state.priceInput) || parseFloat(state.currentPrice) || parseFloat(state.prices[state.currentToken?.tokenAddress] || 0);
        if (!price || price <= 0) {
          throw new Error('Enter a valid price');
        }

        await API.createOrder({
          symbol: state.currentToken.symbol || 'UNKNOWN',
          token_address: state.currentToken.tokenAddress,
          side: state.tradeSide, // BUY or SELL
          order_type: state.activeTradeTab === 'STOP' ? 'STOP_LOSS' : 'LIMIT',
          target_price: price,
          amount: amountVal
        });
        showStatus('Order placed successfully', 'success');
      }

      await refreshData();
    } catch (err) {
      showStatus(err.message, 'error');
    } finally {
      if (btn) {
        btn.disabled = false;
        render(); // Re-render to restore button text
      }
    }
  }

  async function closePositionForToken(amount = null) {
    // Find open trade for current token
    const trade = findTradeForCurrentToken();
    if (trade) {
      const result = await API.closeTrade(trade.trade_id, amount);
      if (result.position_size < 0.000001) { // Fully closed
        showStatus(`Closed ${result.symbol}: ${Utils.formatPnL(result.pnl)}`, result.pnl >= 0 ? 'success' : 'error');
      } else {
        showStatus(`Sold Partial ${result.symbol}: ${Utils.formatPnL(result.pnl)}`, 'success');
      }
    } else {
      showStatus('No open position to sell', 'error');
    }
  }
  async function handleCloseTrade(tradeId) {
    const btn = document.querySelector(`[data-close-trade="${tradeId}"]`);
    if (btn) {
      btn.disabled = true;
      btn.textContent = '...';
    }

    try {
      const result = await API.closeTrade(tradeId);
      showStatus(`Closed ${result.symbol}: ${Utils.formatPnL(result.pnl)}`,
        result.pnl >= 0 ? 'success' : 'error');
      await refreshData();
    } catch (err) {
      showStatus(err.message, 'error');
      if (btn) {
        btn.disabled = false;
        btn.textContent = 'CLOSE';
      }
    }
  }

  async function handleCancelOrder(orderId) {
    try {
      await API.cancelOrder(orderId);
      showStatus('Order cancelled', 'success');
      await refreshData();
    } catch (err) {
      showStatus(err.message, 'error');
    }
  }

  function showStatus(text, type = 'success') {
    state.statusMessage = { text, type };
    render();
    setTimeout(() => {
      state.statusMessage = null;
      render();
    }, 3000);
  }

  function saveWatchlist() {
    if (typeof chrome !== 'undefined' && chrome.storage) {
      chrome.storage.local.set({ watchlist: state.watchlist });
    }
  }

  // ============ Data Loading ============
  async function refreshData() {
    try {
      const [balanceData, trades, orders, history] = await Promise.all([
        API.getBalance().catch(() => null),
        API.getOpenTrades().catch(() => []),
        API.getOrders('PENDING').catch(() => []),
        API.getTradeHistory(10).catch(() => []),
      ]);

      if (balanceData) {
        state.balance = balanceData.current_balance || balanceData.balance || 0;
      }
      state.openTrades = Array.isArray(trades) ? trades : [];
      state.orders = Array.isArray(orders) ? orders : [];
      state.history = Array.isArray(history) ? history : [];
      state.loading = false;

      render();
      API.updateBadge().catch(() => { });

      // Subscribe open positions for real-time price updates via background WS
      if (state.openTrades.length > 0) {
        const mints = [...new Set(state.openTrades.map(t => t.token_address).filter(Boolean))];
        if (mints.length > 0) {
          chrome.runtime.sendMessage({ type: 'SUBSCRIBE_PRICES', mints }).catch(() => { });
        }
        // Also do a one-shot fetch as safety net
        fetchPricesForPositions();
      }
    } catch (err) {
      console.error('[FAFO] Data refresh error:', err);
      state.loading = false;
      render();
    }
  }

  async function fetchPricesForPositions() {
    // Get unique token addresses from open trades
    const addresses = [...new Set(state.openTrades.map(t => t.token_address).filter(Boolean))];
    if (addresses.length === 0) return;

    try {
      // Route through background worker (no CORS restrictions)
      chrome.runtime.sendMessage(
        { type: 'FETCH_PRICES', addresses },
        (response) => {
          if (chrome.runtime.lastError) return;
          if (response && response.success && response.prices) {
            let updated = false;
            for (const [addr, price] of Object.entries(response.prices)) {
              if (price && price > 0) {
                state.prices[addr] = price;
                updated = true;
              }
            }
            if (updated) {
              updatePositionsPnl();
            }
          }
        }
      );
    } catch (e) {
      // Price fetch failed, ignore
    }
  }

  async function checkAuth() {
    try {
      const auth = await API.getAuthState();
      state.loggedIn = auth.loggedIn;
      state.username = auth.username;

      if (auth.loggedIn) {
        await refreshData();
      } else {
        state.loading = false;
        render();
      }
    } catch (err) {
      console.error('[FAFO] Auth check error:', err);
      state.loading = false;
      render();
    }
  }

  // ============ Trade Matching ============
  function findTradeForCurrentToken() {
    if (!state.currentToken || !state.openTrades.length) return null;
    const addr = state.currentToken.tokenAddress?.toLowerCase();

    // 1. Exact address match
    if (addr) {
      const exact = state.openTrades.find(t => t.token_address?.toLowerCase() === addr);
      if (exact) return exact;
    }

    // 2. Symbol match (case-insensitive)
    const sym = state.currentToken.symbol?.toUpperCase();
    if (sym && sym !== '???' && sym.length > 2) {
      const bySymbol = state.openTrades.find(t => t.symbol?.toUpperCase() === sym);
      if (bySymbol) return bySymbol;
    }

    return null;
  }

  // ============ Platform Detection ============
  function detectToken() {
    const detection = PlatformDetector.detect(window.location.href);
    state.detection = detection;

    if (detection?.tokenInfo) {
      const symbol = PlatformDetector.getSymbol(detection);
      state.currentToken = {
        ...detection.tokenInfo,
        symbol: symbol || detection.tokenInfo.symbol || null,
      };

      // Try to refine token address from page (e.g. DexScreener pair -> mint)
      if (detection.getTokenMintFromPage) {
        setTimeout(() => {
          const mint = detection.getTokenMintFromPage();
          if (mint && mint !== state.currentToken.tokenAddress) {
            state.currentToken.tokenAddress = mint;
            render();
          }
        }, 1000);
      }

      // Retry symbol detection if we don't have one yet (dynamic pages like Jupiter, Telemetry)
      if (!state.currentToken.symbol && detection.getSymbolFromPage) {
        const retrySymbol = (attempts) => {
          if (attempts <= 0) return;
          setTimeout(() => {
            const sym = detection.getSymbolFromPage();
            if (sym) {
              state.currentToken.symbol = sym;
              render();
            } else {
              retrySymbol(attempts - 1);
            }
          }, 1500);
        };
        retrySymbol(10); // Try up to 10 times over ~15 seconds (SPA pages)
      }

      // Try to resolve Pair -> Token via API (DexScreener)
      if (detection.resolve) {
        detection.resolve(state.currentToken.tokenAddress).then(resolved => {
          if (resolved && resolved !== state.currentToken.tokenAddress) {
            console.log('[FAFO] Resolved Pair Address to Mint:', resolved);
            state.currentToken.tokenAddress = resolved;
            render();
          }
        });
      }

      // Initial price check
      detectPrice();

      // Periodic price scraping for SPA pages (price may load late)
      if (detection.getPriceFromPage) {
        let priceRetries = 0;
        const priceInterval = setInterval(() => {
          detectPrice();
          priceRetries++;
          if (priceRetries > 20 || state.currentPrice > 0) {
            clearInterval(priceInterval);
          }
        }, 2000);
      }
    }

    // Relay token detection to background (for side panel)
    if (state.currentToken) {
      chrome.runtime.sendMessage({
        type: 'TOKEN_DETECTED',
        token: state.currentToken,
        price: state.currentPrice || 0,
      }).catch(() => { });
    }
  }

  function detectPrice() {
    let price = null;

    // 1. Try DOM scraping (platform-specific)
    if (state.detection && state.detection.getPriceFromPage) {
      price = state.detection.getPriceFromPage();
    }

    // 2. Fallback: WebSocket price from backend
    if (!price && state.currentToken?.tokenAddress) {
      price = state.prices[state.currentToken.tokenAddress] || null;
    }

    // 3. Update state if we got a valid price
    if (price && price > 0 && price !== state.currentPrice) {
      state.currentPrice = price;
      updatePositionsPnl();

      // Relay scraped price to background (for side panel)
      chrome.runtime.sendMessage({
        type: 'PRICE_SCRAPED',
        price: price,
        mint: state.currentToken?.tokenAddress,
      }).catch(() => { });

      // Rerender only if quick trade is visible
      const previewEl = document.getElementById('fafo-price-preview');
      if (previewEl) {
        updatePricePreview();
      }
    }
  }

  // WebSocket is now handled by background.js (no CSP restrictions)
  // Price updates arrive via chrome.runtime.onMessage 'PRICE_UPDATE'

  function updatePositionsPnl() {
    const pnlElements = document.querySelectorAll(`[data-pnl-trade-id]`);
    pnlElements.forEach(el => {
      const tradeId = el.dataset.pnlTradeId;
      const trade = state.openTrades.find(t => t.trade_id === tradeId);

      if (!trade) return;

      let price = null;

      // 1. WebSocket Price
      if (trade.token_address && state.prices[trade.token_address]) {
        price = state.prices[trade.token_address];
      }

      // 2. Fallback to Active Page Price
      if (!price) {
        const currentTokenAddr = state.currentToken?.tokenAddress?.toLowerCase();
        if (state.currentPrice && currentTokenAddr) {
          if (trade.token_address.toLowerCase() === currentTokenAddr) {
            price = state.currentPrice;
          }
          else if (state.currentToken.symbol && trade.symbol &&
            trade.symbol.toUpperCase() === state.currentToken.symbol.toUpperCase()) {
            price = state.currentPrice;
          }
        }
      }

      if (price) {
        const entry = parseFloat(el.dataset.entry);
        const size = parseFloat(el.dataset.size);

        const pnl = (price - entry) * size;
        const pnlPct = ((price - entry) / entry) * 100;

        el.textContent = Utils.formatPnL(pnl);
        el.className = `fafo-position-pnl-value fafo-pnl ${pnl >= 0 ? 'positive' : 'negative'}`;

        const pctEl = document.querySelector(`[data-pct-trade-id="${tradeId}"]`);
        if (pctEl) {
          pctEl.textContent = Utils.formatPercent(pnlPct);
          pctEl.className = `fafo-position-pnl-amount ${pnl >= 0 ? 'positive' : 'negative'}`;
        }
      }
    });

    // Also update the balance bar's unrealized PnL
    const unrealizedEl = document.getElementById('fafo-unrealized-pnl');
    if (unrealizedEl) {
      const totalPnl = calcTotalUnrealizedPnl();
      unrealizedEl.textContent = Utils.formatPnL(totalPnl);
      unrealizedEl.className = `fafo-pnl ${totalPnl >= 0 ? 'positive' : 'negative'}`;
      unrealizedEl.id = 'fafo-unrealized-pnl';
    }
  }

  function updatePricePreview() {
    const isBuy = state.tradeSide === 'BUY';
    const input = document.getElementById('fafo-trade-amount');
    const val = parseFloat(input?.value || 0);
    const price = state.currentPrice;
    const previewEl = document.getElementById('fafo-price-preview');

    if (!previewEl || !price) return;

    if (isBuy) {
      // Input is USD -> Show Tokens
      const tokens = val / price;
      previewEl.textContent = `≈ ${tokens.toFixed(6)} ${state.currentToken?.symbol || 'Tokens'}`;
    } else {
      // Input is Tokens -> Show USD
      const usd = val * price;
      previewEl.textContent = `≈ ${Utils.formatUSD(usd)}`;
    }
  }

  // ============ URL Change Observer (SPA navigation) ============
  function observeUrlChanges() {
    let lastUrl = window.location.href;
    let lastTitle = document.title;

    function onUrlChange() {
      if (window.location.href !== lastUrl) {
        lastUrl = window.location.href;
        console.log('[FAFO] SPA URL changed:', lastUrl);
        state.currentPrice = null; // Reset price for new token
        detectToken();
        render();
      }
    }

    function onTitleChange() {
      const newTitle = document.title;
      if (newTitle !== lastTitle && newTitle !== 'TELEMETRY terminal') {
        lastTitle = newTitle;
        // Title changed → token info may have updated, retry symbol
        if (state.detection && state.detection.getSymbolFromPage) {
          const sym = state.detection.getSymbolFromPage();
          if (sym && sym !== state.currentToken?.symbol) {
            state.currentToken.symbol = sym;
            render();
          }
        }
      }
    }

    // MutationObserver for SPA navigation (DOM changes)
    const observer = new MutationObserver(() => {
      onUrlChange();
      onTitleChange();
    });

    observer.observe(document.body, { childList: true, subtree: true });

    // Also observe <title> changes directly
    const titleEl = document.querySelector('title');
    if (titleEl) {
      const titleObserver = new MutationObserver(() => {
        onTitleChange();
        onUrlChange();
      });
      titleObserver.observe(titleEl, { childList: true, characterData: true, subtree: true });
    }

    // Listen for popstate
    window.addEventListener('popstate', () => {
      detectToken();
      render();
    });

    // History API override — pushState
    const pushState = history.pushState;
    history.pushState = function () {
      pushState.apply(history, arguments);
      setTimeout(() => {
        detectToken();
        render();
      }, 100);
    };

    // History API override — replaceState (used by many SPAs like Telemetry)
    const replaceState = history.replaceState;
    history.replaceState = function () {
      replaceState.apply(history, arguments);
      setTimeout(() => {
        onUrlChange();
      }, 100);
    };

    // Interval-based URL polling fallback (catches all edge cases)
    setInterval(() => {
      onUrlChange();
    }, 1000);
  }

  // ============ Message Listener ============
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === 'AUTH_STATE_CHANGED') {
      state.loggedIn = message.loggedIn;
      state.username = message.username;
      if (message.loggedIn) {
        refreshData();
        // Grab token
        chrome.storage.local.get(['authToken'], (result) => {
          if (result.authToken) {
            state.token = result.authToken;
          }
        });
      } else {
        state.openTrades = [];
        state.orders = [];
        state.history = [];
        state.balance = 0;
        state.token = null;
        state.prices = {};
        render();
      }
    }

    if (message.type === 'URL_CHANGED') {
      detectToken();
      render();
    }

    // Toggle panel visibility (from extension icon click)
    if (message.type === 'TOGGLE_PANEL') {
      state.panelHidden = !state.panelHidden;
      render();
    }

    // Real-time price updates from background WebSocket
    if (message.type === 'PRICE_UPDATE') {
      if (message.mint && message.price > 0) {
        state.prices[message.mint] = message.price;

        // Also update currentPrice if this mint matches the active token
        if (state.currentToken?.tokenAddress === message.mint) {
          state.currentPrice = message.price;
          updatePricePreview();
        }

        updatePositionsPnl();
      }
    }

    // Listen for Token/Auth updates
    if (message.type === 'TOKEN_UPDATED') {
      state.token = message.token;
    }

    // Panel mode switching
    if (message.type === 'PANEL_MODE_CHANGED') {
      state.panelMode = message.mode;
      const panel = document.getElementById('fafo-paper-panel');
      if (panel) {
        if (message.mode === 'sidepanel') {
          panel.style.display = 'none';
        } else {
          panel.style.display = 'flex';
        }
      }
    }
  });

  // ============ Initialize ============
  async function init() {
    // Check stored panel mode
    const stored = await chrome.storage.local.get(['panelMode']);
    if (stored.panelMode) state.panelMode = stored.panelMode;
    detectToken();
    createPanel();

    // If side panel mode is active, hide the floating panel
    if (state.panelMode === 'sidepanel') {
      const panel = document.getElementById('fafo-panel');
      if (panel) panel.style.display = 'none';
    }

    render();
    checkAuth();
    observeUrlChanges();

    // Load Watchlist + Auth Token
    try {
      chrome.storage.local.get(['watchlist', 'authToken'], (result) => {
        if (result.watchlist) {
          state.watchlist = result.watchlist;
          render();
        }
        if (result.authToken) {
          state.token = result.authToken;
          // WebSocket handled by background.js automatically
        }
      });
    } catch (e) {
      // Storage access might fail in some contexts, ignore
    }

    // Refresh data every 5 seconds (Real-time)
    setInterval(() => {
      if (state.loggedIn) refreshData();
    }, 5000);

    // Fast Price Polling (1s)
    setInterval(detectPrice, 1000);
  }

  // Wait for DOM to be ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
