/**
 * API Client for Chrome Extension
 * 
 * All requests go through the background service worker
 * to avoid CORS issues from content scripts.
 */

const API = (() => {
    /**
     * Send a message to the background service worker
     */
    function sendMessage(msg) {
        return new Promise((resolve, reject) => {
            chrome.runtime.sendMessage(msg, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                    return;
                }
                resolve(response);
            });
        });
    }

    /**
     * Make an API request via background worker
     */
    async function request(endpoint, method = 'GET', body = null) {
        const result = await sendMessage({
            type: 'API_REQUEST',
            endpoint,
            method,
            body,
        });

        if (!result.success) {
            throw new Error(result.error || 'API request failed');
        }
        return result.data;
    }

    // ============ Auth ============

    async function login(username, password) {
        return sendMessage({ type: 'LOGIN', username, password });
    }

    async function register(username, password, email) {
        return sendMessage({ type: 'REGISTER', username, password, email });
    }

    async function logout() {
        return sendMessage({ type: 'LOGOUT' });
    }

    async function getAuthState() {
        return sendMessage({ type: 'GET_AUTH_STATE' });
    }

    async function setApiUrl(url) {
        return sendMessage({ type: 'SET_API_URL', url });
    }

    async function getApiUrl() {
        return sendMessage({ type: 'GET_API_URL' });
    }

    // ============ Trading ============

    async function getBalance() {
        return request('/auth/balance');
    }

    async function getCurrentUser() {
        return request('/auth/me');
    }

    async function openTrade(symbol, tokenAddress, amountUsd) {
        return request('/trades/open', 'POST', {
            symbol,
            token_address: tokenAddress,
            amount_usd: amountUsd,
        });
    }

    async function closeTrade(tradeId, amount = null) {
        return request(`/trades/${tradeId}/close`, 'POST', amount ? { amount } : null);
    }

    async function getOpenTrades() {
        return request('/trades/open');
    }

    async function getTradeHistory(limit = 20) {
        return request(`/trades/history?limit=${limit}`);
    }

    // ============ Orders ============

    async function createOrder(orderData) {
        return request('/orders', 'POST', orderData);
    }

    async function getOrders(status = null) {
        const query = status ? `?status=${status}` : '';
        return request(`/orders${query}`);
    }

    async function cancelOrder(orderId) {
        return request(`/orders/${orderId}`, 'DELETE');
    }

    async function updateBadge() {
        return sendMessage({ type: 'UPDATE_BADGE' });
    }

    return {
        login, register, logout, getAuthState, setApiUrl, getApiUrl,
        getBalance, getCurrentUser,
        openTrade, closeTrade, getOpenTrades, getTradeHistory,
        createOrder, getOrders, cancelOrder,
        updateBadge,
    };
})();

if (typeof window !== 'undefined') {
    window.API = API;
}
