/**
 * Utility functions for FAFO Paper Trading Extension
 */

const Utils = (() => {
    /**
     * Format USD amount
     */
    function formatUSD(amount) {
        if (amount == null || isNaN(amount)) return '$0.00';
        const num = Number(amount);
        if (Math.abs(num) >= 1000) {
            return '$' + num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
        }
        return '$' + num.toFixed(2);
    }

    /**
     * Format price (handle very small prices for meme coins)
     */
    function formatPrice(price) {
        if (price == null || isNaN(price)) return '$0';
        const num = Number(price);
        if (num === 0) return '$0';
        if (num < 0.00001) return '$' + num.toExponential(4);
        if (num < 0.01) return '$' + num.toFixed(8);
        if (num < 1) return '$' + num.toFixed(6);
        if (num < 100) return '$' + num.toFixed(4);
        return '$' + num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    }

    /**
     * Format P&L with color indicator
     */
    function formatPnL(pnl) {
        const num = Number(pnl);
        const sign = num >= 0 ? '+' : '';
        return sign + formatUSD(num);
    }

    /**
     * Calculate P&L percentage
     */
    function pnlPercent(entryPrice, currentPrice) {
        if (!entryPrice || entryPrice === 0) return 0;
        return ((currentPrice - entryPrice) / entryPrice) * 100;
    }

    /**
     * Format percentage
     */
    function formatPercent(pct) {
        const num = Number(pct);
        const sign = num >= 0 ? '+' : '';
        return sign + num.toFixed(2) + '%';
    }

    /**
     * Format market cap (K/M/B)
     */
    function formatMCap(mcap) {
        if (!mcap || isNaN(mcap)) return null;
        const num = Number(mcap);
        if (num >= 1e9) return '$' + (num / 1e9).toFixed(2) + 'B';
        if (num >= 1e6) return '$' + (num / 1e6).toFixed(2) + 'M';
        if (num >= 1e3) return '$' + (num / 1e3).toFixed(1) + 'K';
        return '$' + num.toFixed(0);
    }

    /**
     * Truncate address for display
     */
    function truncateAddress(address) {
        if (!address || address.length < 10) return address || '';
        return address.slice(0, 6) + '...' + address.slice(-4);
    }

    function timeAgo(dateStr) {
        if (!dateStr) return '';
        // Handle Python datetime format: "2026-02-17 04:55:00" (space instead of T)
        let parsed = dateStr;
        if (typeof parsed === 'string' && !parsed.includes('T')) {
            parsed = parsed.replace(' ', 'T');
        }
        // Append Z if no timezone info to treat as UTC
        if (typeof parsed === 'string' && !parsed.endsWith('Z') && !parsed.includes('+') && !/\d{2}:\d{2}$/.test(parsed.slice(-6))) {
            parsed += 'Z';
        }
        const date = new Date(parsed);
        if (isNaN(date.getTime())) return '';
        const now = new Date();
        const seconds = Math.floor((now - date) / 1000);

        if (seconds < 0) return 'just now';
        if (seconds < 60) return 'just now';
        if (seconds < 3600) return Math.floor(seconds / 60) + 'm ago';
        if (seconds < 86400) return Math.floor(seconds / 3600) + 'h ago';
        if (seconds < 604800) return Math.floor(seconds / 86400) + 'd ago';
        return date.toLocaleDateString();
    }

    /**
     * Create HTML element with attributes
     */
    function createElement(tag, attrs = {}, children = []) {
        const el = document.createElement(tag);
        for (const [key, value] of Object.entries(attrs)) {
            if (key === 'className') el.className = value;
            else if (key === 'textContent') el.textContent = value;
            else if (key === 'innerHTML') el.innerHTML = value;
            else if (key.startsWith('on')) el.addEventListener(key.slice(2).toLowerCase(), value);
            else el.setAttribute(key, value);
        }
        for (const child of children) {
            if (typeof child === 'string') el.appendChild(document.createTextNode(child));
            else if (child) el.appendChild(child);
        }
        return el;
    }

    /**
     * Debounce function
     */
    function debounce(fn, delay) {
        let timer;
        return (...args) => {
            clearTimeout(timer);
            timer = setTimeout(() => fn(...args), delay);
        };
    }

    return {
        formatUSD, formatPrice, formatPnL, pnlPercent,
        formatPercent, formatMCap, truncateAddress, timeAgo,
        createElement, debounce,
    };
})();

if (typeof window !== 'undefined') {
    window.Utils = Utils;
}
