/**
 * FAFO LABS Paper Trading - Popup Script
 * 
 * Handles login, dashboard display, and settings.
 */

document.addEventListener('DOMContentLoaded', init);

// ============ State ============
let currentState = {
    loggedIn: false,
    username: null,
    balance: 0,
    initialBalance: 0,
    openTrades: [],
    orders: [],
    totalTrades: 0,
};

// ============ DOM Elements ============
const views = {
    loading: document.getElementById('loading-view'),
    login: document.getElementById('login-view'),
    register: document.getElementById('register-view'),
    dashboard: document.getElementById('dashboard-view'),
};

// ============ Init ============
let refreshInterval = null;

async function init() {
    // Check auth state
    const auth = await sendMsg({ type: 'GET_AUTH_STATE' });

    if (auth?.loggedIn) {
        currentState.loggedIn = true;
        currentState.username = auth.username;
        showView('dashboard');
        loadDashboard();
        // Auto-refresh every 5 seconds while popup is open
        refreshInterval = setInterval(loadDashboard, 5000);
    } else {
        showView('login');
    }

    // Attach event listeners
    attachListeners();
}

// ============ Communication ============
function sendMsg(msg) {
    return new Promise((resolve) => {
        chrome.runtime.sendMessage(msg, (response) => {
            if (chrome.runtime.lastError) {
                resolve({ error: chrome.runtime.lastError.message });
                return;
            }
            resolve(response);
        });
    });
}

async function apiRequest(endpoint, method = 'GET', body = null) {
    const result = await sendMsg({
        type: 'API_REQUEST',
        endpoint,
        method,
        body,
    });
    if (!result?.success) {
        throw new Error(result?.error || 'Request failed');
    }
    return result.data;
}

// ============ View Management ============
function showView(name) {
    Object.values(views).forEach(v => v.classList.add('hidden'));
    views[name]?.classList.remove('hidden');
}

// ============ Event Listeners ============
function attachListeners() {
    // Login form
    document.getElementById('login-form').addEventListener('submit', handleLogin);

    // Register form
    document.getElementById('register-form').addEventListener('submit', handleRegister);

    // View toggles
    document.getElementById('show-register').addEventListener('click', () => {
        showView('register');
    });
    document.getElementById('show-login').addEventListener('click', () => {
        showView('login');
    });

    // Logout
    document.getElementById('logout-btn').addEventListener('click', handleLogout);

    // Toggle panel on current page
    document.getElementById('toggle-panel-btn')?.addEventListener('click', async () => {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab?.id) {
            chrome.tabs.sendMessage(tab.id, { type: 'TOGGLE_PANEL' });
            // Close popup after toggling
            window.close();
        }
    });

    // Refresh
    document.getElementById('refresh-btn').addEventListener('click', loadDashboard);
}

// ============ Login ============
async function handleLogin(e) {
    e.preventDefault();

    const username = document.getElementById('username').value.trim();
    const password = document.getElementById('password').value.trim();
    const loginBtn = document.getElementById('login-btn');
    const errorEl = document.getElementById('login-error');

    if (!username || !password) {
        showError('Please enter username and password');
        return;
    }

    // Show loading state
    loginBtn.disabled = true;
    document.getElementById('login-text').textContent = 'Signing in...';
    document.getElementById('login-spinner').classList.remove('hidden');
    errorEl.classList.add('hidden');

    try {
        const result = await sendMsg({ type: 'LOGIN', username, password });

        if (result?.success) {
            currentState.loggedIn = true;
            currentState.username = result.data.username || username;
            showToast('Welcome back, ' + currentState.username + '!', 'success');
            showView('dashboard');
            loadDashboard();
            refreshInterval = setInterval(loadDashboard, 5000);
        } else {
            showError(result?.error || 'Login failed');
        }
    } catch (err) {
        showError(err.message);
    } finally {
        loginBtn.disabled = false;
        document.getElementById('login-text').textContent = 'Sign In';
        document.getElementById('login-spinner').classList.add('hidden');
    }
}

function showError(msg) {
    const errorEl = document.getElementById('login-error');
    errorEl.textContent = msg;
    errorEl.classList.remove('hidden');
}

// ============ Register ============
async function handleRegister(e) {
    e.preventDefault();

    const username = document.getElementById('reg-username').value.trim();
    const email = document.getElementById('reg-email').value.trim();
    const password = document.getElementById('reg-password').value;
    const confirm = document.getElementById('reg-confirm').value;
    const registerBtn = document.getElementById('register-btn');
    const errorEl = document.getElementById('register-error');
    const successEl = document.getElementById('register-success');

    errorEl.classList.add('hidden');
    successEl.classList.add('hidden');

    // Validation
    if (!username || !password) {
        showRegisterError('Please fill in all required fields');
        return;
    }
    if (username.length < 3) {
        showRegisterError('Username must be at least 3 characters');
        return;
    }
    if (password.length < 6) {
        showRegisterError('Password must be at least 6 characters');
        return;
    }
    if (password !== confirm) {
        showRegisterError('Passwords do not match');
        return;
    }

    // Show loading state
    registerBtn.disabled = true;
    document.getElementById('register-text').textContent = 'Creating...';
    document.getElementById('register-spinner').classList.remove('hidden');

    try {
        const result = await sendMsg({ type: 'REGISTER', username, password, email: email || null });

        if (result?.success) {
            successEl.textContent = '✓ Account created! Waiting for admin approval.';
            successEl.classList.remove('hidden');
            errorEl.classList.add('hidden');
            document.getElementById('register-form').reset();
            showToast('Account created successfully!', 'success');
            // Auto-switch to login after 2s
            setTimeout(() => showView('login'), 2500);
        } else {
            showRegisterError(result?.error || 'Registration failed');
        }
    } catch (err) {
        showRegisterError(err.message);
    } finally {
        registerBtn.disabled = false;
        document.getElementById('register-text').textContent = 'Create Account';
        document.getElementById('register-spinner').classList.add('hidden');
    }
}

function showRegisterError(msg) {
    const errorEl = document.getElementById('register-error');
    errorEl.textContent = msg;
    errorEl.classList.remove('hidden');
    document.getElementById('register-success').classList.add('hidden');
}



// ============ Logout ============
async function handleLogout() {
    if (refreshInterval) clearInterval(refreshInterval);
    refreshInterval = null;
    await sendMsg({ type: 'LOGOUT' });
    currentState = { loggedIn: false, username: null, balance: 0, initialBalance: 0, openTrades: [], orders: [], totalTrades: 0 };
    showView('login');
    showToast('Logged out', 'success');
}

// ============ Dashboard ============
async function loadDashboard() {
    document.getElementById('dash-username').textContent = '@' + (currentState.username || 'user');

    try {
        // Fetch all data in parallel
        const [balanceData, trades, orders, history] = await Promise.all([
            apiRequest('/auth/me').catch(() => null),
            apiRequest('/trades/open').catch(() => []),
            apiRequest('/orders?status=PENDING').catch(() => []),
            apiRequest('/trades/history?limit=50').catch(() => []),
        ]);

        // Update state
        if (balanceData) {
            currentState.balance = Number(balanceData.current_balance || 0);
            currentState.initialBalance = Number(balanceData.initial_balance || 10000);
        }
        currentState.openTrades = Array.isArray(trades) ? trades : [];
        currentState.orders = Array.isArray(orders) ? orders : [];
        currentState.totalTrades = Array.isArray(history) ? history.length : 0;

        // Fetch live prices for open positions from Jupiter
        if (currentState.openTrades.length > 0) {
            await fetchLivePrices();
        }

        // Update UI
        updateDashboardUI();
        updateServerStatus(true);

        // Update badge
        sendMsg({ type: 'UPDATE_BADGE' });
    } catch (err) {
        console.error('Dashboard load error:', err);
        updateServerStatus(false, err.message);
    }
}

async function fetchLivePrices() {
    const addresses = [...new Set(
        currentState.openTrades
            .map(t => t.token_address)
            .filter(Boolean)
    )];

    if (addresses.length === 0) return;

    try {
        const result = await sendMsg({ type: 'FETCH_PRICES', addresses });
        if (result?.success && result.prices) {
            currentState.livePrices = result.prices;

            // Calculate unrealized PnL for each trade
            for (const trade of currentState.openTrades) {
                const livePrice = currentState.livePrices[trade.token_address];
                if (livePrice && trade.entry_price && trade.quantity) {
                    const entryPrice = Number(trade.entry_price);
                    const quantity = Number(trade.quantity);
                    const currentValue = livePrice * quantity;
                    const costBasis = Number(trade.cost_usd || (entryPrice * quantity));
                    trade._livePnl = currentValue - costBasis;
                    trade._livePrice = livePrice;
                    trade._pnlPercent = entryPrice > 0
                        ? ((livePrice - entryPrice) / entryPrice) * 100
                        : 0;
                }
            }
        }
    } catch (e) {
        console.warn('Failed to fetch live prices:', e);
    }
}

function updateDashboardUI() {
    const { balance, initialBalance, openTrades, orders, totalTrades } = currentState;

    // Calculate unrealized PnL from live prices
    let unrealizedPnl = 0;
    for (const trade of openTrades) {
        if (trade._livePnl != null) {
            unrealizedPnl += trade._livePnl;
        }
    }

    // Balance
    document.getElementById('balance-amount').textContent = formatUSD(balance);

    // Total P&L = realized (balance change) + unrealized (live positions)
    const realizedPnl = balance - initialBalance;
    const totalPnl = realizedPnl + unrealizedPnl;
    const pnlEl = document.getElementById('total-pnl');
    pnlEl.textContent = (totalPnl >= 0 ? '+' : '') + formatUSD(totalPnl);
    pnlEl.className = 'pnl-value ' + (totalPnl > 0 ? 'positive' : totalPnl < 0 ? 'negative' : 'neutral');

    // Balance bar (account for unrealized too)
    const effectiveBalance = balance + unrealizedPnl;
    const pct = initialBalance > 0 ? Math.min((effectiveBalance / initialBalance) * 100, 200) : 0;
    document.getElementById('balance-bar-fill').style.width = Math.min(pct, 100) + '%';

    // Stats
    document.getElementById('stat-positions').textContent = openTrades.length;
    document.getElementById('stat-orders').textContent = orders.length;
    document.getElementById('stat-trades').textContent = totalTrades + openTrades.length;

    // Positions list
    const listEl = document.getElementById('positions-list');
    if (openTrades.length === 0) {
        listEl.innerHTML = '<div class="empty-state">No open positions</div>';
    } else {
        listEl.innerHTML = openTrades.map(trade => {
            const pnl = trade._livePnl != null ? trade._livePnl : Number(trade.pnl || 0);
            const pnlPct = trade._pnlPercent != null ? trade._pnlPercent : 0;
            const pnlClass = pnl >= 0 ? 'positive' : 'negative';
            const pctStr = pnlPct !== 0 ? ` (${pnlPct >= 0 ? '+' : ''}${pnlPct.toFixed(1)}%)` : '';
            return `
        <div class="position-item">
          <div>
            <div class="position-symbol">${trade.symbol}</div>
            <div class="position-size">${formatUSD(trade.cost_usd)}</div>
          </div>
          <div class="position-pnl pnl-value ${pnlClass}">
            ${pnl >= 0 ? '+' : ''}${formatUSD(pnl)}${pctStr}
          </div>
        </div>
      `;
        }).join('');
    }
}

function updateServerStatus(connected, errorMsg) {
    const dot = document.querySelector('.status-dot');
    const text = document.getElementById('server-status');

    if (connected) {
        dot.className = 'status-dot active';
        text.textContent = 'Connected to server';
    } else {
        dot.className = 'status-dot error';
        text.textContent = errorMsg || 'Connection error';
    }
}

// ============ Utils ============
function formatUSD(amount) {
    if (amount == null || isNaN(amount)) return '$0.00';
    const num = Number(amount);
    return '$' + num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
}

function showToast(message, type = 'success') {
    // Remove existing toast
    document.querySelectorAll('.toast').forEach(t => t.remove());

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => toast.remove(), 3000);
}
